<?php
class TaskMaster_DB {
    public static function install() {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();

        $projects = $wpdb->prefix . 'tm_projects';
        $tasks = $wpdb->prefix . 'tm_tasks';

        $sql = "
        CREATE TABLE $projects (
            id BIGINT AUTO_INCREMENT PRIMARY KEY,
            title VARCHAR(255) NOT NULL,
            description TEXT,
            start_date DATE,
            end_date DATE,
            status VARCHAR(50) DEFAULT 'active'
        ) $charset_collate;

        CREATE TABLE $tasks (
            id BIGINT AUTO_INCREMENT PRIMARY KEY,
            project_id BIGINT,
            title VARCHAR(255) NOT NULL,
            description TEXT,
            assigned_to BIGINT,
            due_date DATE,
            status VARCHAR(50) DEFAULT 'pending',
            priority VARCHAR(20) DEFAULT 'normal',
            FOREIGN KEY (project_id) REFERENCES $projects(id)
        ) $charset_collate;
        ";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }

    // افزودن پروژه جدید
    public static function insert_project($title, $description, $start_date, $end_date, $status = 'active') {
        global $wpdb;
        $projects = $wpdb->prefix . 'tm_projects';
        return $wpdb->insert($projects, [
            'title' => $title,
            'description' => $description,
            'start_date' => $start_date,
            'end_date' => $end_date,
            'status' => $status
        ]);
    }

    // دریافت همه پروژه‌ها
    public static function get_projects() {
        global $wpdb;
        $projects = $wpdb->prefix . 'tm_projects';
        return $wpdb->get_results("SELECT * FROM $projects ORDER BY id DESC");
    }

    // حذف پروژه بر اساس ID
    public static function delete_project($id) {
        global $wpdb;
        $projects = $wpdb->prefix . 'tm_projects';
        return $wpdb->delete($projects, ['id' => intval($id)]);
    }

    // دریافت پروژه بر اساس ID
    public static function get_project_by_id($id) {
        global $wpdb;
        $projects = $wpdb->prefix . 'tm_projects';
        return $wpdb->get_row($wpdb->prepare("SELECT * FROM $projects WHERE id = %d", $id));
    }

    // ویرایش پروژه
    public static function update_project($id, $data) {
        global $wpdb;
        $projects = $wpdb->prefix . 'tm_projects';
        return $wpdb->update($projects, $data, ['id' => intval($id)]);
    }

    // افزودن تسک جدید
    public static function insert_task($project_id, $title, $description, $assigned_to, $due_date, $status = 'pending', $priority = 'normal') {
        global $wpdb;
        $tasks = $wpdb->prefix . 'tm_tasks';
        return $wpdb->insert($tasks, [
            'project_id' => $project_id,
            'title' => $title,
            'description' => $description,
            'assigned_to' => $assigned_to,
            'due_date' => $due_date,
            'status' => $status,
            'priority' => $priority
        ]);
    }

    // دریافت تسک‌های یک پروژه
    public static function get_tasks_by_project($project_id) {
        global $wpdb;
        $tasks = $wpdb->prefix . 'tm_tasks';
        return $wpdb->get_results($wpdb->prepare("SELECT * FROM $tasks WHERE project_id = %d ORDER BY id DESC", $project_id));
    }

    // حذف تسک بر اساس ID
    public static function delete_task($id) {
        global $wpdb;
        $tasks = $wpdb->prefix . 'tm_tasks';
        return $wpdb->delete($tasks, ['id' => intval($id)]);
    }

    // ویرایش تسک
    public static function update_task($id, $data) {
        global $wpdb;
        $tasks = $wpdb->prefix . 'tm_tasks';
        return $wpdb->update($tasks, $data, ['id' => intval($id)]);
    }

    // جستجو و فیلتر پروژه‌ها
    public static function search_projects($search = '', $status = '', $date_from = '', $date_to = '') {
        global $wpdb;
        $projects = $wpdb->prefix . 'tm_projects';
        $where = [];
        $params = [];
        
        if (!empty($search)) {
            $where[] = "(title LIKE %s OR description LIKE %s)";
            $params[] = '%' . $wpdb->esc_like($search) . '%';
            $params[] = '%' . $wpdb->esc_like($search) . '%';
        }
        
        if (!empty($status)) {
            $where[] = "status = %s";
            $params[] = $status;
        }
        
        if (!empty($date_from)) {
            $where[] = "start_date >= %s";
            $params[] = $date_from;
        }
        
        if (!empty($date_to)) {
            $where[] = "end_date <= %s";
            $params[] = $date_to;
        }
        
        $sql = "SELECT * FROM $projects";
        if (!empty($where)) {
            $sql .= " WHERE " . implode(' AND ', $where);
        }
        $sql .= " ORDER BY id DESC";
        
        if (!empty($params)) {
            return $wpdb->get_results($wpdb->prepare($sql, $params));
        } else {
            return $wpdb->get_results($sql);
        }
    }

    // جستجو و فیلتر تسک‌ها
    public static function search_tasks($project_id = null, $search = '', $status = '', $assigned_to = '', $priority = '', $date_from = '', $date_to = '') {
        global $wpdb;
        $tasks = $wpdb->prefix . 'tm_tasks';
        $where = [];
        $params = [];
        
        if ($project_id) {
            $where[] = "project_id = %d";
            $params[] = $project_id;
        }
        
        if (!empty($search)) {
            $where[] = "(title LIKE %s OR description LIKE %s)";
            $params[] = '%' . $wpdb->esc_like($search) . '%';
            $params[] = '%' . $wpdb->esc_like($search) . '%';
        }
        
        if (!empty($status)) {
            $where[] = "status = %s";
            $params[] = $status;
        }
        
        if (!empty($assigned_to)) {
            $where[] = "assigned_to = %d";
            $params[] = $assigned_to;
        }
        
        if (!empty($priority)) {
            $where[] = "priority = %s";
            $params[] = $priority;
        }
        
        if (!empty($date_from)) {
            $where[] = "due_date >= %s";
            $params[] = $date_from;
        }
        
        if (!empty($date_to)) {
            $where[] = "due_date <= %s";
            $params[] = $date_to;
        }
        
        $sql = "SELECT * FROM $tasks";
        if (!empty($where)) {
            $sql .= " WHERE " . implode(' AND ', $where);
        }
        $sql .= " ORDER BY id DESC";
        
        if (!empty($params)) {
            return $wpdb->get_results($wpdb->prepare($sql, $params));
        } else {
            return $wpdb->get_results($sql);
        }
    }

    // دریافت تسک‌ها برای تقویم (بر اساس بازه زمانی)
    public static function get_tasks_for_calendar($start_date = '', $end_date = '') {
        global $wpdb;
        $tasks = $wpdb->prefix . 'tm_tasks';
        $projects = $wpdb->prefix . 'tm_projects';
        
        $sql = "SELECT t.*, p.title as project_title FROM $tasks t 
                LEFT JOIN $projects p ON t.project_id = p.id";
        
        $where = [];
        $params = [];
        
        if (!empty($start_date)) {
            $where[] = "t.due_date >= %s";
            $params[] = $start_date;
        }
        
        if (!empty($end_date)) {
            $where[] = "t.due_date <= %s";
            $params[] = $end_date;
        }
        
        if (!empty($where)) {
            $sql .= " WHERE " . implode(' AND ', $where);
        }
        
        $sql .= " ORDER BY t.due_date ASC";
        
        if (!empty($params)) {
            return $wpdb->get_results($wpdb->prepare($sql, $params));
        } else {
            return $wpdb->get_results($sql);
        }
    }

    // گزارش آمار پروژه‌ها
    public static function get_project_stats($status = '', $date_from = '', $date_to = '') {
        global $wpdb;
        $projects = $wpdb->prefix . 'tm_projects';
        
        $where = [];
        $params = [];
        
        if (!empty($status)) {
            $where[] = "status = %s";
            $params[] = $status;
        }
        
        if (!empty($date_from)) {
            $where[] = "start_date >= %s";
            $params[] = $date_from;
        }
        
        if (!empty($date_to)) {
            $where[] = "end_date <= %s";
            $params[] = $date_to;
        }
        
        $sql = "SELECT status, COUNT(*) as count FROM $projects";
        if (!empty($where)) {
            $sql .= " WHERE " . implode(' AND ', $where);
        }
        $sql .= " GROUP BY status";
        
        if (!empty($params)) {
            return $wpdb->get_results($wpdb->prepare($sql, $params));
        } else {
            return $wpdb->get_results($sql);
        }
    }

    // گزارش آمار تسک‌ها
    public static function get_task_stats($status = '', $assigned_to = '', $priority = '', $date_from = '', $date_to = '') {
        global $wpdb;
        $tasks = $wpdb->prefix . 'tm_tasks';
        
        $where = [];
        $params = [];
        
        if (!empty($status)) {
            $where[] = "status = %s";
            $params[] = $status;
        }
        
        if (!empty($assigned_to)) {
            $where[] = "assigned_to = %d";
            $params[] = $assigned_to;
        }
        
        if (!empty($priority)) {
            $where[] = "priority = %s";
            $params[] = $priority;
        }
        
        if (!empty($date_from)) {
            $where[] = "due_date >= %s";
            $params[] = $date_from;
        }
        
        if (!empty($date_to)) {
            $where[] = "due_date <= %s";
            $params[] = $date_to;
        }
        
        $sql = "SELECT status, priority, COUNT(*) as count FROM $tasks";
        if (!empty($where)) {
            $sql .= " WHERE " . implode(' AND ', $where);
        }
        $sql .= " GROUP BY status, priority";
        
        if (!empty($params)) {
            return $wpdb->get_results($wpdb->prepare($sql, $params));
        } else {
            return $wpdb->get_results($sql);
        }
    }

    // گزارش عملکرد کاربران
    public static function get_user_performance($date_from = '', $date_to = '') {
        global $wpdb;
        $tasks = $wpdb->prefix . 'tm_tasks';
        
        $where = [];
        $params = [];
        
        if (!empty($date_from)) {
            $where[] = "due_date >= %s";
            $params[] = $date_from;
        }
        
        if (!empty($date_to)) {
            $where[] = "due_date <= %s";
            $params[] = $date_to;
        }
        
        $sql = "SELECT assigned_to, status, COUNT(*) as count FROM $tasks WHERE assigned_to > 0";
        if (!empty($where)) {
            $sql .= " AND " . implode(' AND ', $where);
        }
        $sql .= " GROUP BY assigned_to, status";
        
        if (!empty($params)) {
            return $wpdb->get_results($wpdb->prepare($sql, $params));
        } else {
            return $wpdb->get_results($sql);
        }
    }
} 