<?php
class TaskMaster_Admin {
    public static function add_admin_menu() {
        add_menu_page(
            'TaskMaster Pro',
            'TaskMaster Pro',
            'manage_options',
            'taskmaster-pro',
            [self::class, 'admin_page'],
            'dashicons-clipboard',
            6
        );
        
        // افزودن زیرمنوی تقویم وظایف
        add_submenu_page(
            'taskmaster-pro',
            'تقویم وظایف',
            'تقویم وظایف',
            'manage_options',
            'taskmaster-calendar',
            [self::class, 'calendar_page']
        );
        
        // افزودن زیرمنوی گزارش‌گیری
        add_submenu_page(
            'taskmaster-pro',
            'گزارش‌گیری',
            'گزارش‌گیری',
            'manage_options',
            'taskmaster-reports',
            [self::class, 'reports_page']
        );
    }

    public static function admin_page() {
        // مدیریت عملیات‌ها
        self::handle_admin_actions();
        
        echo '<div class="wrap">';
        echo '<h1>TaskMaster Pro</h1>';
        
        // فرم ثبت پروژه
        self::display_project_form();
        
        // لیست پروژه‌ها
        self::display_projects_list();
        
        // نمایش تسک‌ها اگر پروژه انتخاب شده باشد
        if (isset($_GET['project_id'])) {
            self::display_project_tasks(intval($_GET['project_id']));
        }
        
        echo '</div>';
    }

    // مدیریت عملیات‌های ادمین - بهینه‌شده
    private static function handle_admin_actions() {
        $action = $_GET['action'] ?? $_POST['action'] ?? '';
        
        if (!$action) return;
        
        switch ($action) {
            case 'delete_project':
                if (isset($_GET['id'])) {
                    $deleted = TaskMaster_DB::delete_project(intval($_GET['id']));
                    $message = $deleted ? 'پروژه با موفقیت حذف شد.' : 'خطا در حذف پروژه.';
                    $type = $deleted ? 'success' : 'error';
                    echo '<div class="notice notice-' . $type . ' is-dismissible"><p>' . $message . '</p></div>';
                }
                break;
                
            case 'add_project':
            case 'update_project':
                self::handle_project_save($action);
                break;
        }
    }

    // مدیریت ذخیره پروژه - بهینه‌شده
    private static function handle_project_save($action) {
        $required_fields = ['tm_title'];
        foreach ($required_fields as $field) {
            if (!isset($_POST[$field]) || empty($_POST[$field])) {
                echo '<div class="notice notice-error is-dismissible"><p>لطفاً تمام فیلدهای ضروری را پر کنید.</p></div>';
                return;
            }
        }
        
        $data = [
            'title' => sanitize_text_field($_POST['tm_title']),
            'description' => sanitize_textarea_field($_POST['tm_description'] ?? ''),
            'start_date' => sanitize_text_field($_POST['tm_start_date'] ?? ''),
            'end_date' => sanitize_text_field($_POST['tm_end_date'] ?? ''),
            'status' => sanitize_text_field($_POST['tm_status'] ?? 'active')
        ];
        
        if ($action === 'add_project') {
            $result = TaskMaster_DB::insert_project($data['title'], $data['description'], $data['start_date'], $data['end_date'], $data['status']);
            $message = $result ? 'پروژه با موفقیت ثبت شد.' : 'خطا در ثبت پروژه.';
        } else {
            $id = intval($_POST['tm_edit_project_id']);
            $result = TaskMaster_DB::update_project($id, $data);
            $message = ($result !== false) ? 'پروژه با موفقیت ویرایش شد.' : 'خطا در ویرایش پروژه.';
        }
        
        $type = ($result !== false) ? 'success' : 'error';
        echo '<div class="notice notice-' . $type . ' is-dismissible"><p>' . $message . '</p></div>';
    }

    // نمایش فرم ثبت پروژه
    private static function display_project_form() {
        $edit_id = $_GET['edit_project'] ?? 0;
        $project = null;
        
        if ($edit_id) {
            $project = TaskMaster_DB::get_project_by_id($edit_id);
        }
        
        echo '<h2>' . ($project ? 'ویرایش پروژه' : 'افزودن پروژه جدید') . '</h2>';
        echo '<form method="post" style="margin-bottom:30px; background:#f9f9f9; padding:20px; border-radius:8px;">';
        echo '<input type="hidden" name="action" value="' . ($project ? 'update_project' : 'add_project') . '" />';
        if ($project) {
            echo '<input type="hidden" name="tm_edit_project_id" value="' . intval($project->id) . '" />';
        }
        
        echo '<div style="display:grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap:15px;">';
        
        // عنوان
        echo '<div>';
        echo '<label>عنوان پروژه:</label><br>';
        echo '<input type="text" name="tm_title" value="' . esc_attr($project->title ?? '') . '" placeholder="عنوان پروژه" required style="width:100%;" />';
        echo '</div>';
        
        // توضیحات
        echo '<div>';
        echo '<label>توضیحات:</label><br>';
        echo '<textarea name="tm_description" placeholder="توضیحات" style="width:100%; height:60px;">' . esc_textarea($project->description ?? '') . '</textarea>';
        echo '</div>';
        
        // تاریخ شروع
        echo '<div>';
        echo '<label>تاریخ شروع:</label><br>';
        echo '<input type="date" name="tm_start_date" value="' . esc_attr($project->start_date ?? '') . '" />';
        echo '</div>';
        
        // تاریخ پایان
        echo '<div>';
        echo '<label>تاریخ پایان:</label><br>';
        echo '<input type="date" name="tm_end_date" value="' . esc_attr($project->end_date ?? '') . '" />';
        echo '</div>';
        
        // وضعیت
        echo '<div>';
        echo '<label>وضعیت:</label><br>';
        echo '<select name="tm_status">';
        $statuses = ["active" => "فعال", "completed" => "تکمیل شده", "paused" => "متوقف"];
        foreach ($statuses as $val => $label) {
            $selected = (($project->status ?? 'active') == $val) ? 'selected' : '';
            echo '<option value="' . esc_attr($val) . '" ' . $selected . '>' . esc_html($label) . '</option>';
        }
        echo '</select>';
        echo '</div>';
        
        echo '</div>';
        
        echo '<div style="margin-top:15px;">';
        echo '<button type="submit" class="button button-primary">' . ($project ? 'ذخیره تغییرات' : 'ثبت پروژه') . '</button>';
        if ($project) {
            echo ' <a href="?page=taskmaster-pro" class="button">انصراف</a>';
        }
        echo '</div>';
        echo '</form>';
    }

    // نمایش لیست پروژه‌ها - بهینه‌شده
    private static function display_projects_list() {
        // دریافت پروژه‌ها با فیلتر
        $search = sanitize_text_field($_GET['project_search'] ?? '');
        $status = sanitize_text_field($_GET['project_status'] ?? '');
        $date_from = sanitize_text_field($_GET['project_date_from'] ?? '');
        $date_to = sanitize_text_field($_GET['project_date_to'] ?? '');
        
        // استفاده از کش برای بهبود سرعت
        $cache_key = 'tm_projects_' . md5($search . $status . $date_from . $date_to);
        $projects = wp_cache_get($cache_key);
        
        if ($projects === false) {
            if ($search || $status || $date_from || $date_to) {
                $projects = TaskMaster_DB::search_projects($search, $status, $date_from, $date_to);
            } else {
                $projects = TaskMaster_DB::get_projects();
            }
            wp_cache_set($cache_key, $projects, '', 300); // کش برای 5 دقیقه
        }
        
        echo '<h2>لیست پروژه‌ها</h2>';
        
        // فرم جستجو
        self::display_project_search_form();
        
        if ($projects && count($projects) > 0) {
            echo '<table class="widefat">';
            echo '<thead><tr><th>عنوان</th><th>توضیحات</th><th>تاریخ شروع</th><th>تاریخ پایان</th><th>وضعیت</th><th>عملیات</th></tr></thead><tbody>';
            
            foreach ($projects as $project) {
                echo '<tr>';
                echo '<td>' . esc_html($project->title) . '</td>';
                echo '<td>' . esc_html($project->description) . '</td>';
                echo '<td>' . esc_html($project->start_date) . '</td>';
                echo '<td>' . esc_html($project->end_date) . '</td>';
                echo '<td>' . esc_html($project->status) . '</td>';
                echo '<td>';
                echo '<a href="?page=taskmaster-pro&edit_project=' . intval($project->id) . '" class="button button-small">ویرایش</a> ';
                echo '<a href="?page=taskmaster-pro&action=delete_project&id=' . intval($project->id) . '" onclick="return confirm(\'آیا مطمئن هستید؟\');" class="button button-small button-link-delete">حذف</a> ';
                echo '<a href="?page=taskmaster-pro&project_id=' . intval($project->id) . '" class="button button-small">مدیریت تسک‌ها</a>';
                echo '</td>';
                echo '</tr>';
            }
            
            echo '</tbody></table>';
        } else {
            echo '<p>هیچ پروژه‌ای یافت نشد.</p>';
        }
    }

    // مدیریت تسک‌ها برای هر پروژه
    public static function display_project_tasks($project_id) {
        $project = TaskMaster_DB::get_project_by_id($project_id);
        if (!$project) return;
        
        echo '<div style="margin-top:20px; padding:20px; background:#f9f9f9; border-radius:8px;">';
        echo '<h3>مدیریت تسک‌های پروژه: ' . esc_html($project->title) . '</h3>';
        
        // مدیریت عملیات تسک‌ها
        self::handle_task_actions($project_id);
        
        // فرم ثبت تسک جدید
        self::display_task_form($project_id);
        
        // لیست تسک‌ها
        self::display_tasks_list($project_id);
        
        echo '</div>';
    }

    // مدیریت عملیات تسک‌ها - بهینه‌شده
    private static function handle_task_actions($project_id) {
        $action = $_GET['task_action'] ?? $_POST['task_action'] ?? '';
        
        if (!$action) return;
        
        switch ($action) {
            case 'delete_task':
                if (isset($_GET['task_id'])) {
                    $deleted = TaskMaster_DB::delete_task(intval($_GET['task_id']));
                    $message = $deleted ? 'تسک با موفقیت حذف شد.' : 'خطا در حذف تسک.';
                    $type = $deleted ? 'success' : 'error';
                    echo '<div class="notice notice-' . $type . ' is-dismissible"><p>' . $message . '</p></div>';
                }
                break;
                
            case 'add_task':
            case 'update_task':
                self::handle_task_save($project_id, $action);
                break;
        }
    }

    // مدیریت ذخیره تسک - بهینه‌شده
    private static function handle_task_save($project_id, $action) {
        $required_fields = ['task_title'];
        foreach ($required_fields as $field) {
            if (!isset($_POST[$field]) || empty($_POST[$field])) {
                echo '<div class="notice notice-error is-dismissible"><p>لطفاً عنوان تسک را وارد کنید.</p></div>';
                return;
            }
        }
        
        $data = [
            'title' => sanitize_text_field($_POST['task_title']),
            'description' => sanitize_textarea_field($_POST['task_description'] ?? ''),
            'due_date' => sanitize_text_field($_POST['task_due_date'] ?? ''),
            'status' => sanitize_text_field($_POST['task_status'] ?? 'pending'),
            'priority' => sanitize_text_field($_POST['task_priority'] ?? 'normal'),
            'assigned_to' => intval($_POST['task_assigned_to'] ?? 0)
        ];
        
        if ($action === 'add_task') {
            $result = TaskMaster_DB::insert_task($project_id, $data['title'], $data['description'], $data['assigned_to'], $data['due_date'], $data['status'], $data['priority']);
            $message = $result ? 'تسک با موفقیت ثبت شد.' : 'خطا در ثبت تسک.';
        } else {
            $task_id = intval($_POST['edit_task_id']);
            $result = TaskMaster_DB::update_task($task_id, $data);
            $message = ($result !== false) ? 'تسک با موفقیت ویرایش شد.' : 'خطا در ویرایش تسک.';
        }
        
        $type = ($result !== false) ? 'success' : 'error';
        echo '<div class="notice notice-' . $type . ' is-dismissible"><p>' . $message . '</p></div>';
    }

    // نمایش فرم تسک
    private static function display_task_form($project_id) {
        $edit_task_id = $_GET['edit_task'] ?? 0;
        $task = null;
        
        if ($edit_task_id) {
            $tasks = TaskMaster_DB::get_tasks_by_project($project_id);
            foreach ($tasks as $t) {
                if ($t->id == $edit_task_id) {
                    $task = $t;
                    break;
                }
            }
        }
        
        echo '<h4>' . ($task ? 'ویرایش تسک' : 'افزودن تسک جدید') . '</h4>';
        echo '<form method="post" style="margin-bottom:15px; background:#fff; padding:15px; border-radius:5px;">';
        echo '<input type="hidden" name="task_action" value="' . ($task ? 'update_task' : 'add_task') . '" />';
        if ($task) {
            echo '<input type="hidden" name="edit_task_id" value="' . intval($task->id) . '" />';
        }
        
        echo '<div style="display:grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap:10px;">';
        
        // عنوان
        echo '<div>';
        echo '<label>عنوان تسک:</label><br>';
        echo '<input type="text" name="task_title" value="' . esc_attr($task->title ?? '') . '" placeholder="عنوان تسک" required style="width:100%;" />';
        echo '</div>';
        
        // توضیحات
        echo '<div>';
        echo '<label>توضیحات:</label><br>';
        echo '<textarea name="task_description" placeholder="توضیحات" style="width:100%; height:50px;">' . esc_textarea($task->description ?? '') . '</textarea>';
        echo '</div>';
        
        // تاریخ سررسید
        echo '<div>';
        echo '<label>تاریخ سررسید:</label><br>';
        echo '<input type="date" name="task_due_date" value="' . esc_attr($task->due_date ?? '') . '" />';
        echo '</div>';
        
        // وضعیت
        echo '<div>';
        echo '<label>وضعیت:</label><br>';
        echo '<select name="task_status">';
        $statuses = ["pending" => "در انتظار", "inprogress" => "در حال انجام", "done" => "انجام شده"];
        foreach ($statuses as $val => $label) {
            $selected = (($task->status ?? 'pending') == $val) ? 'selected' : '';
            echo '<option value="' . esc_attr($val) . '" ' . $selected . '>' . esc_html($label) . '</option>';
        }
        echo '</select>';
        echo '</div>';
        
        // اولویت
        echo '<div>';
        echo '<label>اولویت:</label><br>';
        echo '<select name="task_priority">';
        $priorities = ["normal" => "عادی", "high" => "بالا", "low" => "کم"];
        foreach ($priorities as $val => $label) {
            $selected = (($task->priority ?? 'normal') == $val) ? 'selected' : '';
            echo '<option value="' . esc_attr($val) . '" ' . $selected . '>' . esc_html($label) . '</option>';
        }
        echo '</select>';
        echo '</div>';
        
        // مسئول
        echo '<div>';
        echo '<label>مسئول:</label><br>';
        $users = get_users(['fields' => ['ID', 'display_name']]);
        echo '<select name="task_assigned_to">';
        echo '<option value="0">بدون مسئول</option>';
        foreach ($users as $user) {
            $selected = (($task->assigned_to ?? 0) == $user->ID) ? 'selected' : '';
            echo '<option value="' . intval($user->ID) . '" ' . $selected . '>' . esc_html($user->display_name) . '</option>';
        }
        echo '</select>';
        echo '</div>';
        
        echo '</div>';
        
        echo '<div style="margin-top:10px;">';
        echo '<button type="submit" class="button button-primary">' . ($task ? 'ذخیره تغییرات' : 'ثبت تسک') . '</button>';
        if ($task) {
            echo ' <a href="?page=taskmaster-pro&view_tasks=' . $project_id . '" class="button">انصراف</a>';
        }
        echo '</div>';
        echo '</form>';
    }

    // نمایش لیست تسک‌ها - بهینه‌شده
    private static function display_tasks_list($project_id) {
        // استفاده از کش برای بهبود سرعت
        $cache_key = 'tm_tasks_' . $project_id;
        $tasks = wp_cache_get($cache_key);
        
        if ($tasks === false) {
            $tasks = TaskMaster_DB::get_tasks_by_project($project_id);
            wp_cache_set($cache_key, $tasks, '', 300); // کش برای 5 دقیقه
        }
        
        if ($tasks && count($tasks) > 0) {
            echo '<table class="widefat">';
            echo '<thead><tr><th>عنوان</th><th>توضیحات</th><th>مسئول</th><th>تاریخ سررسید</th><th>وضعیت</th><th>اولویت</th><th>عملیات</th></tr></thead><tbody>';
            
            // کش کردن اطلاعات کاربران
            $user_cache = [];
            
            foreach ($tasks as $task) {
                // استفاده از کش برای اطلاعات کاربر
                if (!isset($user_cache[$task->assigned_to])) {
                    $user_cache[$task->assigned_to] = $task->assigned_to ? get_userdata($task->assigned_to) : null;
                }
                $assigned = $user_cache[$task->assigned_to];
                
                echo '<tr>';
                echo '<td>' . esc_html($task->title) . '</td>';
                echo '<td>' . esc_html($task->description) . '</td>';
                echo '<td>' . ($assigned ? esc_html($assigned->display_name) : '-') . '</td>';
                echo '<td>' . esc_html($task->due_date) . '</td>';
                echo '<td>' . esc_html($task->status) . '</td>';
                echo '<td>' . esc_html($task->priority) . '</td>';
                echo '<td>';
                echo '<a href="?page=taskmaster-pro&project_id=' . $project_id . '&edit_task=' . intval($task->id) . '" class="button button-small">ویرایش</a> ';
                echo '<a href="?page=taskmaster-pro&project_id=' . $project_id . '&task_action=delete_task&task_id=' . intval($task->id) . '" onclick="return confirm(\'آیا مطمئن هستید؟\');" class="button button-small button-link-delete">حذف</a>';
                echo '</td>';
                echo '</tr>';
            }
            
            echo '</tbody></table>';
        } else {
            echo '<p>هیچ تسکی برای این پروژه ثبت نشده است.</p>';
        }
    }

    // نمایش فرم جستجوی پروژه‌ها
    private static function display_project_search_form() {
        echo '<div style="background:#f9f9f9; padding:15px; margin-bottom:20px; border-radius:5px;">';
        echo '<form method="get" style="display:flex; gap:10px; flex-wrap:wrap; align-items:center;">';
        echo '<input type="hidden" name="page" value="taskmaster-pro" />';
        echo '<input type="text" name="project_search" placeholder="جستجو در عنوان و توضیحات" value="' . esc_attr($_GET['project_search'] ?? '') . '" style="width:200px;" /> ';
        echo '<select name="project_status">';
        echo '<option value="">همه وضعیت‌ها</option>';
        $statuses = ["active" => "فعال", "completed" => "تکمیل شده", "paused" => "متوقف"];
        foreach ($statuses as $val => $label) {
            $selected = (isset($_GET['project_status']) && $_GET['project_status'] == $val) ? 'selected' : '';
            echo '<option value="' . esc_attr($val) . '" ' . $selected . '>' . esc_html($label) . '</option>';
        }
        echo '</select> ';
        echo '<input type="date" name="project_date_from" placeholder="از تاریخ" value="' . esc_attr($_GET['project_date_from'] ?? '') . '" /> ';
        echo '<input type="date" name="project_date_to" placeholder="تا تاریخ" value="' . esc_attr($_GET['project_date_to'] ?? '') . '" /> ';
        echo '<button type="submit" class="button">جستجو</button> ';
        echo '<a href="?page=taskmaster-pro" class="button">پاک کردن فیلتر</a>';
        echo '</form>';
        echo '</div>';
    }

    public static function enqueue_assets() {
        // فقط در صفحات افزونه بارگذاری شود
        $screen = get_current_screen();
        if (!$screen || strpos($screen->id, 'taskmaster') === false) {
            return;
        }
        
        // بارگذاری استایل‌ها
        wp_enqueue_style('taskmaster-admin-style', plugin_dir_url(__FILE__) . '../assets/css/admin.css', [], '1.0.0');
        
        // بارگذاری اسکریپت‌ها
        wp_enqueue_script('taskmaster-admin-script', plugin_dir_url(__FILE__) . '../assets/js/admin.js', ['jquery'], '1.0.0', true);
        
        // انتقال متغیرهای PHP به JavaScript
        wp_localize_script('taskmaster-admin-script', 'taskmaster_ajax', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('taskmaster_nonce'),
            'strings' => [
                'confirm_delete' => 'آیا مطمئن هستید؟',
                'loading' => 'در حال بارگذاری...',
                'error' => 'خطا در عملیات'
            ]
        ]);
    }

    // صفحه تقویم وظایف
    public static function calendar_page() {
        // تعیین ماه و سال فعلی
        $current_month = isset($_GET['month']) ? intval($_GET['month']) : date('n');
        $current_year = isset($_GET['year']) ? intval($_GET['year']) : date('Y');
        
        // محاسبه اولین و آخرین روز ماه
        $first_day = mktime(0, 0, 0, $current_month, 1, $current_year);
        $last_day = mktime(0, 0, 0, $current_month + 1, 0, $current_year);
        
        // دریافت تسک‌های ماه جاری
        $start_date = date('Y-m-d', $first_day);
        $end_date = date('Y-m-d', $last_day);
        $tasks = TaskMaster_DB::get_tasks_for_calendar($start_date, $end_date);
        
        // گروه‌بندی تسک‌ها بر اساس تاریخ
        $tasks_by_date = [];
        foreach ($tasks as $task) {
            $date = $task->due_date;
            if (!isset($tasks_by_date[$date])) {
                $tasks_by_date[$date] = [];
            }
            $tasks_by_date[$date][] = $task;
        }
        
        echo '<div class="wrap">';
        echo '<h1>تقویم وظایف</h1>';
        
        // ناوبری ماه
        $prev_month = $current_month - 1;
        $prev_year = $current_year;
        if ($prev_month < 1) {
            $prev_month = 12;
            $prev_year--;
        }
        
        $next_month = $current_month + 1;
        $next_year = $current_year;
        if ($next_month > 12) {
            $next_month = 1;
            $next_year++;
        }
        
        echo '<div style="margin-bottom:20px;">';
        echo '<a href="?page=taskmaster-calendar&month=' . $prev_month . '&year=' . $prev_year . '" class="button">&laquo; ماه قبل</a> ';
        echo '<span style="font-size:18px; font-weight:bold; margin:0 20px;">' . date('F Y', $first_day) . '</span>';
        echo '<a href="?page=taskmaster-calendar&month=' . $next_month . '&year=' . $next_year . '" class="button">ماه بعد &raquo;</a>';
        echo '</div>';
        
        // نمایش تقویم
        echo '<table style="width:100%; border-collapse:collapse; border:1px solid #ddd;">';
        
        // هدر روزهای هفته
        $weekdays = ['یکشنبه', 'دوشنبه', 'سه‌شنبه', 'چهارشنبه', 'پنج‌شنبه', 'جمعه', 'شنبه'];
        echo '<tr style="background:#f5f5f5;">';
        foreach ($weekdays as $day) {
            echo '<th style="border:1px solid #ddd; padding:10px; text-align:center;">' . $day . '</th>';
        }
        echo '</tr>';
        
        // محاسبه روز شروع هفته
        $first_day_of_week = date('w', $first_day);
        $first_day_of_week = ($first_day_of_week + 1) % 7; // تبدیل به تقویم شمسی
        
        // نمایش روزهای ماه
        $current_day = 1;
        $days_in_month = date('t', $first_day);
        
        for ($week = 0; $week < 6; $week++) {
            echo '<tr>';
            for ($day_of_week = 0; $day_of_week < 7; $day_of_week++) {
                if (($week == 0 && $day_of_week < $first_day_of_week) || $current_day > $days_in_month) {
                    echo '<td style="border:1px solid #ddd; padding:10px; min-height:100px; background:#f9f9f9;"></td>';
                } else {
                    $current_date = date('Y-m-d', mktime(0, 0, 0, $current_month, $current_day, $current_year));
                    $is_today = ($current_date == date('Y-m-d')) ? 'background:#e7f3ff;' : '';
                    
                    echo '<td style="border:1px solid #ddd; padding:10px; min-height:100px; vertical-align:top; ' . $is_today . '">';
                    echo '<div style="font-weight:bold; margin-bottom:5px;">' . $current_day . '</div>';
                    
                    // نمایش تسک‌های این روز
                    if (isset($tasks_by_date[$current_date])) {
                        foreach ($tasks_by_date[$current_date] as $task) {
                            $status_color = '';
                            switch ($task->status) {
                                case 'done': $status_color = 'background:#d4edda; color:#155724;'; break;
                                case 'inprogress': $status_color = 'background:#fff3cd; color:#856404;'; break;
                                case 'pending': $status_color = 'background:#f8d7da; color:#721c24;'; break;
                            }
                            
                            echo '<div style="margin:2px 0; padding:3px; border-radius:3px; font-size:11px; ' . $status_color . '">';
                            echo '<strong>' . esc_html($task->title) . '</strong><br>';
                            echo '<small>پروژه: ' . esc_html($task->project_title) . '</small>';
                            echo '</div>';
                        }
                    }
                    
                    echo '</td>';
                    $current_day++;
                }
            }
            echo '</tr>';
        }
        
        echo '</table>';
        
        // راهنمای رنگ‌ها
        echo '<div style="margin-top:20px; padding:10px; background:#f9f9f9; border-radius:5px;">';
        echo '<h3>راهنمای رنگ‌ها:</h3>';
        echo '<div style="display:flex; gap:20px; flex-wrap:wrap;">';
        echo '<span style="padding:5px 10px; background:#f8d7da; color:#721c24; border-radius:3px;">در انتظار</span>';
        echo '<span style="padding:5px 10px; background:#fff3cd; color:#856404; border-radius:3px;">در حال انجام</span>';
        echo '<span style="padding:5px 10px; background:#d4edda; color:#155724; border-radius:3px;">انجام شده</span>';
        echo '</div>';
        echo '</div>';
        
        echo '</div>';
    }

    // صفحه گزارش‌گیری
    public static function reports_page() {
        // دریافت پارامترهای فیلتر
        $report_type = sanitize_text_field($_GET['report_type'] ?? 'projects');
        $date_from = sanitize_text_field($_GET['date_from'] ?? '');
        $date_to = sanitize_text_field($_GET['date_to'] ?? '');
        $status = sanitize_text_field($_GET['status'] ?? '');
        $assigned_to = sanitize_text_field($_GET['assigned_to'] ?? '');
        
        echo '<div class="wrap">';
        echo '<h1>گزارش‌گیری TaskMaster Pro</h1>';
        
        // فرم فیلتر گزارش‌ها
        echo '<div style="background:#f9f9f9; padding:15px; margin-bottom:20px; border-radius:5px;">';
        echo '<h3>فیلتر گزارش‌ها</h3>';
        echo '<form method="get" style="display:flex; gap:10px; flex-wrap:wrap; align-items:center;">';
        echo '<input type="hidden" name="page" value="taskmaster-reports" />';
        echo '<select name="report_type">';
        echo '<option value="projects"' . ($report_type == 'projects' ? ' selected' : '') . '>گزارش پروژه‌ها</option>';
        echo '<option value="tasks"' . ($report_type == 'tasks' ? ' selected' : '') . '>گزارش تسک‌ها</option>';
        echo '<option value="users"' . ($report_type == 'users' ? ' selected' : '') . '>گزارش عملکرد کاربران</option>';
        echo '</select> ';
        echo '<input type="date" name="date_from" placeholder="از تاریخ" value="' . esc_attr($date_from) . '" /> ';
        echo '<input type="date" name="date_to" placeholder="تا تاریخ" value="' . esc_attr($date_to) . '" /> ';
        echo '<select name="status">';
        echo '<option value="">همه وضعیت‌ها</option>';
        $statuses = ["active" => "فعال", "completed" => "تکمیل شده", "paused" => "متوقف"];
        foreach ($statuses as $val => $label) {
            $selected = ($status == $val) ? 'selected' : '';
            echo '<option value="' . esc_attr($val) . '" ' . $selected . '>' . esc_html($label) . '</option>';
        }
        echo '</select> ';
        if ($report_type == 'tasks' || $report_type == 'users') {
            $users = get_users(['fields' => ['ID', 'display_name']]);
            echo '<select name="assigned_to">';
            echo '<option value="">همه کاربران</option>';
            foreach ($users as $u) {
                $selected = ($assigned_to == $u->ID) ? 'selected' : '';
                echo '<option value="' . intval($u->ID) . '" ' . $selected . '>' . esc_html($u->display_name) . '</option>';
            }
            echo '</select> ';
        }
        echo '<button type="submit" class="button">نمایش گزارش</button> ';
        echo '<a href="?page=taskmaster-reports" class="button">پاک کردن فیلتر</a>';
        echo '</form>';
        echo '</div>';
        
        // نمایش گزارش بر اساس نوع
        switch ($report_type) {
            case 'projects':
                self::display_project_report($date_from, $date_to, $status);
                break;
            case 'tasks':
                self::display_task_report($date_from, $date_to, $status, $assigned_to);
                break;
            case 'users':
                self::display_user_report($date_from, $date_to);
                break;
        }
        
        echo '</div>';
    }

    // نمایش گزارش پروژه‌ها
    private static function display_project_report($date_from, $date_to, $status) {
        $stats = TaskMaster_DB::get_project_stats($status, $date_from, $date_to);
        $total_projects = 0;
        $status_counts = [];
        
        foreach ($stats as $stat) {
            $total_projects += $stat->count;
            $status_counts[$stat->status] = $stat->count;
        }
        
        echo '<h2>گزارش پروژه‌ها</h2>';
        echo '<div style="display:grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap:20px; margin-bottom:30px;">';
        
        // کارت آمار کلی
        echo '<div style="background:#fff; padding:20px; border-radius:8px; box-shadow:0 2px 4px rgba(0,0,0,0.1);">';
        echo '<h3>آمار کلی</h3>';
        echo '<p><strong>کل پروژه‌ها:</strong> ' . $total_projects . '</p>';
        echo '<p><strong>پروژه‌های فعال:</strong> ' . ($status_counts['active'] ?? 0) . '</p>';
        echo '<p><strong>پروژه‌های تکمیل شده:</strong> ' . ($status_counts['completed'] ?? 0) . '</p>';
        echo '<p><strong>پروژه‌های متوقف:</strong> ' . ($status_counts['paused'] ?? 0) . '</p>';
        echo '</div>';
        
        // نمودار وضعیت پروژه‌ها
        echo '<div style="background:#fff; padding:20px; border-radius:8px; box-shadow:0 2px 4px rgba(0,0,0,0.1);">';
        echo '<h3>نمودار وضعیت پروژه‌ها</h3>';
        echo '<div style="height:200px; display:flex; align-items:end; gap:10px; margin-top:20px;">';
        
        $max_count = max($status_counts);
        foreach ($status_counts as $status_name => $count) {
            $height = $max_count > 0 ? ($count / $max_count) * 150 : 0;
            $color = $status_name == 'active' ? '#28a745' : ($status_name == 'completed' ? '#17a2b8' : '#dc3545');
            echo '<div style="display:flex; flex-direction:column; align-items:center; flex:1;">';
            echo '<div style="width:40px; height:' . $height . 'px; background:' . $color . '; border-radius:3px;"></div>';
            echo '<small style="margin-top:5px;">' . $count . '</small>';
            echo '<small style="font-size:10px;">' . ucfirst($status_name) . '</small>';
            echo '</div>';
        }
        echo '</div>';
        echo '</div>';
        echo '</div>';
        
        // دکمه دانلود گزارش
        echo '<div style="margin-top:20px;">';
        echo '<a href="?page=taskmaster-reports&report_type=projects&download=pdf&date_from=' . $date_from . '&date_to=' . $date_to . '&status=' . $status . '" class="button button-primary">دانلود PDF</a> ';
        echo '<a href="?page=taskmaster-reports&report_type=projects&download=excel&date_from=' . $date_from . '&date_to=' . $date_to . '&status=' . $status . '" class="button button-secondary">دانلود Excel</a>';
        echo '</div>';
    }

    // نمایش گزارش تسک‌ها
    private static function display_task_report($date_from, $date_to, $status, $assigned_to) {
        $stats = TaskMaster_DB::get_task_stats($status, $assigned_to, '', $date_from, $date_to);
        $total_tasks = 0;
        $status_counts = [];
        $priority_counts = [];
        
        foreach ($stats as $stat) {
            $total_tasks += $stat->count;
            $status_counts[$stat->status] = ($status_counts[$stat->status] ?? 0) + $stat->count;
            $priority_counts[$stat->priority] = ($priority_counts[$stat->priority] ?? 0) + $stat->count;
        }
        
        echo '<h2>گزارش تسک‌ها</h2>';
        echo '<div style="display:grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap:20px; margin-bottom:30px;">';
        
        // کارت آمار کلی
        echo '<div style="background:#fff; padding:20px; border-radius:8px; box-shadow:0 2px 4px rgba(0,0,0,0.1);">';
        echo '<h3>آمار کلی</h3>';
        echo '<p><strong>کل تسک‌ها:</strong> ' . $total_tasks . '</p>';
        echo '<p><strong>در انتظار:</strong> ' . ($status_counts['pending'] ?? 0) . '</p>';
        echo '<p><strong>در حال انجام:</strong> ' . ($status_counts['inprogress'] ?? 0) . '</p>';
        echo '<p><strong>انجام شده:</strong> ' . ($status_counts['done'] ?? 0) . '</p>';
        echo '</div>';
        
        // نمودار اولویت‌ها
        echo '<div style="background:#fff; padding:20px; border-radius:8px; box-shadow:0 2px 4px rgba(0,0,0,0.1);">';
        echo '<h3>نمودار اولویت‌ها</h3>';
        echo '<div style="height:200px; display:flex; align-items:end; gap:10px; margin-top:20px;">';
        
        $max_count = max($priority_counts);
        $colors = ['high' => '#dc3545', 'normal' => '#ffc107', 'low' => '#28a745'];
        foreach ($priority_counts as $priority => $count) {
            $height = $max_count > 0 ? ($count / $max_count) * 150 : 0;
            $color = $colors[$priority] ?? '#6c757d';
            echo '<div style="display:flex; flex-direction:column; align-items:center; flex:1;">';
            echo '<div style="width:40px; height:' . $height . 'px; background:' . $color . '; border-radius:3px;"></div>';
            echo '<small style="margin-top:5px;">' . $count . '</small>';
            echo '<small style="font-size:10px;">' . ucfirst($priority) . '</small>';
            echo '</div>';
        }
        echo '</div>';
        echo '</div>';
        echo '</div>';
        
        // دکمه دانلود گزارش
        echo '<div style="margin-top:20px;">';
        echo '<a href="?page=taskmaster-reports&report_type=tasks&download=pdf&date_from=' . $date_from . '&date_to=' . $date_to . '&status=' . $status . '&assigned_to=' . $assigned_to . '" class="button button-primary">دانلود PDF</a> ';
        echo '<a href="?page=taskmaster-reports&report_type=tasks&download=excel&date_from=' . $date_from . '&date_to=' . $date_to . '&status=' . $status . '&assigned_to=' . $assigned_to . '" class="button button-secondary">دانلود Excel</a>';
        echo '</div>';
    }

    // نمایش گزارش عملکرد کاربران
    private static function display_user_report($date_from, $date_to) {
        $stats = TaskMaster_DB::get_user_performance($date_from, $date_to);
        $user_stats = [];
        
        foreach ($stats as $stat) {
            if (!isset($user_stats[$stat->assigned_to])) {
                $user_stats[$stat->assigned_to] = [];
            }
            $user_stats[$stat->assigned_to][$stat->status] = $stat->count;
        }
        
        echo '<h2>گزارش عملکرد کاربران</h2>';
        echo '<div style="background:#fff; padding:20px; border-radius:8px; box-shadow:0 2px 4px rgba(0,0,0,0.1); margin-bottom:20px;">';
        echo '<table class="widefat">';
        echo '<thead><tr><th>کاربر</th><th>تسک‌های انجام شده</th><th>تسک‌های در حال انجام</th><th>تسک‌های در انتظار</th><th>کل تسک‌ها</th></tr></thead><tbody>';
        
        foreach ($user_stats as $user_id => $statuses) {
            $user = get_userdata($user_id);
            if ($user) {
                $done = $statuses['done'] ?? 0;
                $inprogress = $statuses['inprogress'] ?? 0;
                $pending = $statuses['pending'] ?? 0;
                $total = $done + $inprogress + $pending;
                
                echo '<tr>';
                echo '<td>' . esc_html($user->display_name) . '</td>';
                echo '<td>' . $done . '</td>';
                echo '<td>' . $inprogress . '</td>';
                echo '<td>' . $pending . '</td>';
                echo '<td><strong>' . $total . '</strong></td>';
                echo '</tr>';
            }
        }
        echo '</tbody></table>';
        echo '</div>';
        
        // دکمه دانلود گزارش
        echo '<div style="margin-top:20px;">';
        echo '<a href="?page=taskmaster-reports&report_type=users&download=pdf&date_from=' . $date_from . '&date_to=' . $date_to . '" class="button button-primary">دانلود PDF</a> ';
        echo '<a href="?page=taskmaster-reports&report_type=users&download=excel&date_from=' . $date_from . '&date_to=' . $date_to . '" class="button button-secondary">دانلود Excel</a>';
        echo '</div>';
    }
} 